#!/usr/bin/env python3

import argparse
import urllib.request, urllib.error, urllib.parse
import json
import traceback

from intigua_client import IntiguaRestClient
import intigua_client

def add_tag_to_servers(server_list):
    try:
        client = IntiguaRestClient(args.user, args.apikey, args.url)
        #########################################
        # [ARUN] Get all exisiting tags
        existing_tags = client.listTags()
        #########################################
        for id, server in server_list.items():
            tag_exists = False
            s = getServer(client, server['name'])
            new_tags = s["tags"]
            #print(s["tags"])
            for existing_tag in s["tags"]:
                if existing_tag.get("name") == server['tag']:
                    tag_exists = True
                    print("Tag '%s' exists in '%s', will not add" % (server['tag'], server['name']))
                    break
            if not tag_exists:
                print("Will add '%s' tag to '%s'" % (server['tag'], server['name']))
                ##############################################
                # [ARUN] NOTE: Call add_tag to create a tag if tag is not yet added in list. 
                # So tag should be added before assign to server.
                ex_tag = [p for p in existing_tags if p['name'] == server['tag']]
                if(not ex_tag):
                    print("Tag ('%s') is not found. Creating it.." % (server['tag']))
                    client.addTag(server['tag'])
                    existing_tags.append({"name": server['tag']})
                ##############################################
                tag_obj = {}
                tag_obj["name"] = server['tag']
                new_tags.append(tag_obj)
                s["tags"] = new_tags
                #print(s["tags"])
                client.updateServer(s)
            ##################################################
            # [ARUN] # check for --createsmartgroup flag and create smartgroup if it is not exists
            if(args.smartgroup):
                smartgroup_name = server['tag'] + " group"
                #Check wheather smartgroup already exists
                existing_smartgroups = client.listSmartGroups()
                is_sg_exisits = [p for p in existing_smartgroups if p['name'] == smartgroup_name]
                if(not is_sg_exisits):
                    print("Creating a Smart group ('%s')" % smartgroup_name)
                    filters = [
                        {
                            "key":"source,tagName",
                            "type":"Tags",
                            "values":[[("INTIGUA,%s" % server['tag']),server['tag']]]
                        }
                    ]
                    sg = {
                        "name": smartgroup_name,
                        "description": "Created for Tag - " + server['tag'],
                        "filters": filters
                    }
                    client.addSmartGroup(sg)
            ##################################################

            
    except urllib.error.HTTPError as e:
        print(e)
        print(traceback.format_exc())
        exit(1)

def getServer(client, server):
    l = client.findServer(server)
    if len(l) != 1:
        raise "Server search did not return a single server (%d servers found). Please change your server indetifier to be more specific."
    return l[0]

def does_server_exist(client,server_name,ip):
    server_exists_by_name = client.findServerByParams(hostname=server_name)
    server_exists_by_ip = client.findServerByParams(ip=ip)
    if not server_exists_by_name and not server_exists_by_ip:
        return 0
    else:
        return 1

def pair(arg):
    # For simplity, assume arg is a pair of integers
    # separated by a comma. If you want to do more
    # validation, raise argparse.ArgumentError if you
    # encounter a problem.
    return [str(x) for x in arg.split(',')]


parser = argparse.ArgumentParser(description='Physical Server Bulk Adder')
parser.add_argument('--list', dest='list', type=pair, nargs='+', help=' Enter Hostname and Tag in the following format: Server,Tag ...', required=True)
parser.add_argument('--url', dest='url', type=str,
                    help='URL to Intigua Server. Please use full rest path. e.g:https://[intigua-servername]/vmanage-server/rest/rest-api',
                    required=True)
parser.add_argument('--user', dest='user', type=str, help='Username', required=True)
parser.add_argument('--apikey', dest='apikey', type=str, help='API Key', required=True)
#########################################
# [ARUN] Added arg for creating smartgroup 
parser.add_argument('--createsmartgroup', dest='smartgroup',action="store_true", help='Flag to create Smart Group', required=False)
#########################################
args = parser.parse_args()


def parse_serverinv_to_dict(serverinv):
    server_list = {}
    try:
        dict_reader = serverinv
        i=0
        for server in dict_reader:
                if server [0] != '' and server[1] != '':
                    server_list[i] = {}
                    server_list[i]['name'] = server[0]
                    server_list[i]['tag'] = server[1]
                    i += 1
                else:
                    print(server[0]+" skipped! -- Missing information")
    except IOError as e:
        print(e)
        exit(1)
    return server_list

def main(): 
    servers_list = args.list
    server_dict = parse_serverinv_to_dict(servers_list)    
    add_tag_to_servers(server_dict)

if __name__ == '__main__':
    main()