#!/usr/bin/env python3

import json, urllib.request, urllib.error, urllib.parse, urllib.request, urllib.parse, urllib.error, urllib.parse
import re
import os
import sys
import time
import ssl
import hashlib

TASK_TYPE = "vnd.intigua.vmanager.Task"
VAGENT_TYPE = "vnd.intigua.vmanager.VAgent"
SERVERS_TYPE = "vnd.intigua.vmanager.Servers"

current_milli_time = lambda: int(round(time.time() * 1000))

experimental_base_url = "/vmanage-server/rest/experimental"


def getObj(response):
    returned_data = response.read()
    content_type = response.headers.get('Content-Type')

    # if we got back json data, return it
    if content_type is not None and returned_data and "json" in content_type.lower():
        return json.loads(returned_data.decode('utf-8'))
    return returned_data


def getLink(obj, rel):
    for link in obj["links"]:
        if link["rel"] == rel:
            return link["href"]


def addLink(obj, rel, link):
    rel_and_link = {"rel": rel, "href": link}
    if not obj["links"]:
        obj["links"] = []
    obj["links"].append(rel_and_link)


def setLink(obj, rel, link):
    if obj["links"]:
        obj["links"] = [l for l in obj["links"] if l["rel"] != rel]
    addLink(obj, rel, link)


def checkForError(response):
    # read the response and check for errors
    errCode = response.getcode()
    if errCode < 200 or errCode > 399:
        raise Exception("Received HTTP code %s" % errCode)


def contactAndHash(*args):
    strToHash = ";".join([x if x else "" for x in args])
    return hashlib.sha256(strToHash).digest().encode("base64").strip()


def checkType(response, objType):
    ct = response.headers.get("Content-Type")
    if (objType.lower() + "+") not in ct.lower():
        raise Exception("Wront type! got %s and expected %s" % (objType, ct))
    return response


def getObjAndEtag(response, objtype):
    checkType(response, objtype)
    return (getObj(response), response.info().get("ETag"))


def getTaskAndEtag(response):
    return getObjAndEtag(response, TASK_TYPE)


def getVAgentAndEtag(response):
    return getObjAndEtag(response, VAGENT_TYPE)


def getFileName(httpHeader):
    return re.findall(r'filename="?(\S+?)"?$', httpHeader)[0]


# http://stackoverflow.com/questions/120951/how-can-i-normalize-a-url-in-python
def url_fix(s, charset='utf-8'):
    """Sometimes you get an URL by a user that just isn't a real
    URL because it contains unsafe characters like ' ' and so on.  This
    function can fix some of the problems in a similar way browsers
    handle data entered by the user:

    :param charset: The target charset for the URL if the url was
                    given as unicode string.
    """
    if isinstance(s, str):
        s = s.encode(charset, 'ignore')
    scheme, netloc, path, qs, anchor = urllib.parse.urlsplit(s)
    path = urllib.parse.quote(path, '/%()!')
    qs = urllib.parse.quote_plus(qs, ':&=')
    return urllib.parse.urlunsplit((scheme, netloc, path, qs, anchor))


class TaskTimeoutException(Exception):
    def __init___(self, task_timeout_exception_args):
        Exception.__init__(self, "task timed out {0}".format(task_timeout_exception_args))
        self.taskExceptionArgs = task_timeout_exception_args


class IntiguaRestClient(object):
    """
    Intigua REST API client.
    A few notes:
     - The only url constructed on the client side is the entryPoint to the API
     - Use of content types to
    """

    def __init__(self, user, apiKey, url=None, debug=False, no_check_certificate=True, profileName=None):

        self.entryPoint = url if url[-1] == '/' else url + "/"
        self.user, self.apiKey = user, apiKey
        self.debug = debug
        self.no_check_certificate = no_check_certificate

        self.opener = None
        self.build_opener()

        self.services = self.getServices()
        self.profileName = profileName

    def getClientUser(self):
        return self.user

    def getEntryPoint(self):
        return self.entryPoint

    def getProfileName(self):
        return self.profileName

    def build_opener(self):
        opener = urllib.request.build_opener(*self.get_handles(False))
        self.opener = opener

    def get_password_mgr(self):
        mgr = urllib.request.HTTPPasswordMgr()
        mgr.add_password("Intigua REST API", urllib.parse.urljoin(self.entryPoint, "/"), self.user, self.apiKey)
        return mgr

    def get_handles(self, debug=False):
        debuglevel = 0
        if debug:
            debuglevel = 1
        handlers = [urllib.request.HTTPDigestAuthHandler(self.get_password_mgr())]
        context = self.get_context()
        if context:
            handlers = handlers + [urllib.request.HTTPSHandler(debuglevel=debuglevel, context=context)]
        if debug:
            handles = [urllib.request.HTTPHandler(debuglevel=1)] + handlers
        return handlers

    def get_context(self):
        if hasattr(ssl, 'create_default_context'):  # Python >= 3.4 or 2.7.9
            context = ssl.create_default_context(ssl.Purpose.SERVER_AUTH)
            if self.no_check_certificate:
                context.check_hostname = False
                context.verify_mode = ssl.CERT_NONE
                return context

    def doUrlQueryGetResponse(self, url, verb="GET", json_data=None, data_content_type=None, headers=None, timeout=300):
        if headers is None:
            headers = {}
        data = None
        # Encode json data object
        if json_data is not None:
            data = json.dumps(json_data)
            data_content_type = 'application/json'

        if data_content_type:
            headers['Content-Type'] = data_content_type

        # create the url
        url = urllib.parse.urljoin(self.entryPoint, url)

        if self.debug:
            print(verb, url)
            if data:
                print("\tBody:", data)

        if data:
            data = data.encode("utf-8")

        # prepare the request
        def request():
            req = urllib.request.Request(url, data, headers=headers)
            req.get_method = lambda: verb

            response = self.opener.open(req, timeout=timeout)
            return response

        for i in range(2):
            try:
                response = request()
                break
            except urllib.error.HTTPError as e:
                # if the error is 401, try again. i think this is a python bug.
                if e.getcode() == 401:
                    if self.debug:
                        print("Digest auth failed, retrying %d" % i)
                    time.sleep(1)
                    # reset urllib2 state, this solves some digest auth issues.
                    self.build_opener()
                    continue

                raise
        else:
            response = request()

        if self.debug:
            print("result code", response.getcode())
        return response

    def doUrlQuery(self, url, verb="GET", json_data=None, data_content_type=None, headers=None):

        response = self.doUrlQueryGetResponse(url, verb, json_data, data_content_type, headers)

        # Decode received data object
        return getObj(response)

    def returnTaskOrObj(self, response):
        # 202 Accepted - means that the server created a task object for the operation
        if response.getcode() == 202:
            # return the task object with the ETAG
            return getTaskAndEtag(response)

        if response.getcode() == 204:
            # Nothing was done. return None
            return

        # in case it is a regular 200, just return the object
        return getObj(response)

    def get(self, url):
        result = self.doUrlQuery(url, verb="GET")
        return result

    def getServices(self):
        services = self.doUrlQuery(self.entryPoint)
        if not isinstance(services, dict):
            raise ValueError(
                "Invalid services - check your URL. it usually looks like this: https://server-addr/vmanage-server/rest/rest-api")
        return services

    def listCredentials(self):
        return self.doUrlQuery(self.services["credentials"])

    def addCredentials(self, accountName, username, password, os):
        cred = {"username": username, "accountname": accountName, "password": password, "type": os.upper()}
        response = self.doUrlQueryGetResponse(self.services["credentials"], "POST", cred)
        checkForError(response)

    def listTags(self):
        tags_url = "/vmanage-server/rest/rest-api/tags"
        return self.doUrlQuery(tags_url)

    def addTag(self, name):
        tags_url = "/vmanage-server/rest/rest-api/tags"
        tag = {"name": name}
        response = self.doUrlQueryGetResponse(tags_url, "POST", tag)
        checkForError(response)
    
    ########################################################
    # [ARUN] For smart group
    def addSmartGroup(self, smartgroup):
        smartgroup_url = experimental_base_url + "/patch-governance/computer-groups"
        body = {
            "name":smartgroup['name'],
            "description":smartgroup['description'],
            "filters": smartgroup['filters']
        }
        response = self.doUrlQueryGetResponse(smartgroup_url, "POST", body)
        checkForError(response)
        
    def listSmartGroups(self):
        smartgroup_url = experimental_base_url + "/patch-governance/computer-groups"
        return self.doUrlQuery(smartgroup_url)
    ########################################################
    
    def deleteTag(self, tag):
        response = self.doUrlQueryGetResponse(tag["url"], "DELETE")
        checkForError(response)

    def findCredentials(self, name):
        creds = [x for x in self.doUrlQuery(self.services["credentials"]) if x["accountname"] == name]
        if len(creds) != 1:
            raise Exception("Problem finding credentials " + str(creds))
        return creds[0]

    def listPackages(self):
        return self.doUrlQuery(self.services["vagents"])

    def addServer(self, server):
        return self.doUrlQuery(self.services["servers"], verb="POST", json_data=server)

    # Thes use of PUT method on a the servers collection in order to merge two servers is deprecated
    def mergeServers(self, serversToMerge):
        return self.doUrlQuery(self.services["servers"], verb="PUT", json_data=serversToMerge)

    #    def updateServer(self, server, state):
    #        return self.doUrlQuery(server["url"], verb="PUT", json_data = state)

    def updateServer(self, server):
        response = self.doUrlQueryGetResponse(server["url"], "PUT", server)

    def removeServer(self, server):
        response = self.doUrlQueryGetResponse(server["url"], verb="DELETE")
        return self.returnTaskOrObj(response)

    def is_server_connected(self, server):
        connector_url = getLink(server, "connector")
        try:
            return self.doUrlQuery(connector_url)
        except urllib.error.HTTPError as e:
            if e.getcode() == 404:
                return None
            if e.getcode() == 500 and "NotFoundRestException" in e.read():
                return None
            raise

    def findAllServers(self):
        return self.doUrlQuery(self.services["servers"])

    def findServer(self, queryString):
        return self.doUrlQuery(self.services["servers"] + "?" + urllib.parse.urlencode({"q": queryString}))

    def findServerByParams(self, **params):
        servers = self.doUrlQuery(self.services["servers"] + "?" + urllib.parse.urlencode(params))
        if "ip" in params:
            # ip requires an exact search:
            servers = [s for s in servers if s.get("ip", None) == params["ip"]]
        return servers

    def getVAgentsInServer(self, server):

        vagentsUrl = getLink(server, "vagents")
        return self.doUrlQuery(vagentsUrl)

    def getConnectorLog(self, server):
        connectorUrl = getLink(server, "connectorUrl")
        response = self.doUrlQueryGetResponse(connectorUrl + "/logs")

        checkForError(response)
        buffer_size = 1024
        with open(getFileName(response.headers.get("Content-Disposition")), "wb") as outf:
            while 1:
                copy_buffer = response.read(buffer_size)
                if copy_buffer:
                    outf.write(copy_buffer)
                else:
                    break

    def getDebugLogs(self):
        response = self.doUrlQueryGetResponse("debug/archive")

        checkForError(response)
        buffer_size = 1024
        with open(getFileName(response.headers.get("Content-Disposition")), "wb") as outf:
            while 1:
                copy_buffer = response.read(buffer_size)
                if copy_buffer:
                    outf.write(copy_buffer)
                else:
                    break

    def changeVAgentState(self, vagent, state, verb="PUT", etag=None):
        vagentUrl = vagent["url"]

        headers = None
        if etag:
            headers = {"If-Match": etag}
        response = self.doUrlQueryGetResponse(vagentUrl, verb, state, headers=headers)

        return self.returnTaskOrObj(response)

    def startVAgent(self, vagent):
        return self.changeVAgentState(vagent, {"state": "RUNNING"})

    def restartVAgent(self, vagent):
        return self.changeVAgentState(vagent, {"action": "RESTART"})

    def registerVAgent(self, vagent):
        registrationUrl = getLink(vagent, "registration")
        response = self.doUrlQueryGetResponse(registrationUrl, "POST", None)

        return self.returnTaskOrObj(response)

    def deregisterVAgent(self, vagent):
        registrationUrl = getLink(vagent, "registration")
        response = self.doUrlQueryGetResponse(registrationUrl, "DELETE", None)

        return self.returnTaskOrObj(response)

    def setGroupsVAgent(self, vagent):
        registrationUrl = getLink(vagent, "registration")
        response = self.doUrlQueryGetResponse(registrationUrl, "PUT", None)

        return self.returnTaskOrObj(response)

    def stopVAgent(self, vagent):
        return self.changeVAgentState(vagent, {"state": "STOPPED"})

    def changeVAgentConfigPackage(self, vagent, configpackage):
        return self.changeVAgentState(vagent, {"configpackage": configpackage})

    def removeVAgent(self, vagent):
        return self.changeVAgentState(vagent, None, "DELETE")

    def refreshServer(self, server):
        return self.doUrlQuery(server["url"])

    def getServer(self, server):
        return self.doUrlQuery(server["url"])

    def deployVAgentToServerTask(self, server, vagentName, vagentVersion, configPackage, configpackageversion=None):
        # post the new task to the vagent url
        # example
        # {"name": "dummy", "version": "0.0.1", "configpackage": "dummy_felix_1", "configpackageversion":"0.1"}
        vagentsUrl = getLink(server, "vagents")
        postParams = {}
        if configpackageversion is not None:
            postParams = {"name": vagentName, "version": vagentVersion, "configpackage": configPackage,
                          "configpackageversion": configpackageversion}
        else:
            postParams = {"name": vagentName, "version": vagentVersion, "configpackage": configPackage}
        response = self.doUrlQueryGetResponse(vagentsUrl, "POST", postParams)

        return self.returnTaskOrObj(response)

    def deployVAgentByUuidTask(self, instanceUuid, vagentName, vagentVersion, configPackage, configpackageversion=None):
        # find the server
        servers = self.findServerByParams(uuid=instanceUuid)

        if len(servers) != 1:
            raise Exception("Invalid number of servers found")

        # get the server found
        server = servers[0]
        return self.deployVAgentToServerTask(server, vagentName, vagentVersion, configPackage, configpackageversion)

    def deployVAgentByHostnameTask(self, hostname, vagentName, vagentVersion, configPackage, configpackageversion=None):
        # find the server
        servers = self.findServerByParams(hostname=hostname)

        if len(servers) != 1:
            raise Exception("Invalid number of servers found")

        # get the server found
        server = servers[0]
        return self.deployVAgentToServerTask(server, vagentName, vagentVersion, configPackage, configpackageversion)

    def defineServerCredentials(self, server, credentials):
        serverUrl = server["url"]
        params = {"credentialsUrl": credentials["url"]}
        response = self.doUrlQueryGetResponse(serverUrl, "PUT", params)

    def getServerCredentials(self, server):
        credsUrl = getLink(server, "credentials")
        if not credsUrl:
            return None
        return self.doUrlQuery(credsUrl)

    def deployConnectorToServerTask(self, server, credentials=None):
        # post the new task to the vagent url
        connectorUrl = getLink(server, "connector")
        params = None
        if credentials:
            params = {"credentialsUrl": credentials["url"]}
        response = self.doUrlQueryGetResponse(connectorUrl, "POST", params)

        return self.returnTaskOrObj(response)

    def changeConnectorState(self, server, op):
        serverUrl = server["url"]

        # post the new task to the vagent url
        connectorUrl = getLink(server, "connector")
        response = self.doUrlQueryGetResponse(connectorUrl, "PUT", {"op": op})

        return self.returnTaskOrObj(response)

    def updateConnector(self, server):
        return self.changeConnectorState(server, "upgrade-connector")

    def updateConnectorLoader(self, server):
        return self.changeConnectorState(server, "upgrade-loader")

    def updateConnectorStatus(self, server):
        return self.changeConnectorState(server, "update-state")

    def runScriptOnServerTask(self, server, script, timeout=600):
        commandUrl = getLink(server, "connector") + "/command"
        with open(script, 'r') as script_fp:
            scriptData = script_fp.read()
        filename = os.path.basename(script)
        response = self.doUrlQueryGetResponse(commandUrl, "POST",
                                              {"content": scriptData, "name": filename, "envVars": {}}, timeout=timeout)

        return self.returnTaskOrObj(response)

    def runScriptOnServerTextTask(self, server, script_name, scriptData, timeout=600):
        commandUrl = getLink(server, "connector") + "/command"
        response = self.doUrlQueryGetResponse(commandUrl, "POST",
                                              {"content": scriptData, "name": script_name, "envVars": {}},
                                              timeout=timeout)

        return self.returnTaskOrObj(response)

    def removeConnectorToServerTask(self, server):
        # post the new task to the vagent url
        connectorUrl = getLink(server, "connector")
        response = self.doUrlQueryGetResponse(connectorUrl, "DELETE")

        return self.returnTaskOrObj(response)

    def waitForTask(self, task, etag=None, timeout=300):
        max_timeout = current_milli_time() + timeout * 1000
        while task["state"] == "inprogress":
            # wait for a second for the task to progress
            time.sleep(1)
            # use ETag if we have it.
            task, etag = self.queryTask(task, etag)
            if max_timeout < current_milli_time():
                raise TaskTimeoutException({'task': task, 'etag': etag, 'timeout': timeout})
        # return the updated task
        return task, etag

    def queryTask(self, task, etag=None, timeout=10):
        headers = {"If-None-Match": etag} if etag else None
        try:
            response = self.doUrlQueryGetResponse(task["url"], "GET", headers=headers, timeout=timeout)
            task, etag = getTaskAndEtag(response)
            return (task, etag)
        except urllib.error.HTTPError as e:
            # if the task was not modified (304) do nothing.
            if e.getcode() == 304:
                # nothing changed, return the existing objects
                return (task, etag)
            raise

    def remediate(self, ruleName, serverName=None):
        body = {}
        body['ruleName'] = ruleName
        if serverName is not None:
            body['serverName'] = serverName

        try:
            response = self.doUrlQueryGetResponse(experimental_base_url + "/compliance/remediation", "POST", body)
            return self.returnTaskOrObj(response)
        except urllib.error.HTTPError as e:
            if e.getcode() == 401:
                return "responded with error code 401: this be a result of non exists rule or server not in fixable sate"
            raise


# short cut, for the lazy (like me!)
_I = IntiguaRestClient

DEFAULT_CONF_FILE = os.path.join(os.path.expanduser('~'), '.intigua_rest_config')
client = None


def main():
    import argparse
    parser = argparse.ArgumentParser(description='Intigua rest client')

    # Do not use the default paramter in the params that are also read from the config file!!
    parser.add_argument('--url', dest='url', type=str,
                        help='The REST end point of the core server (usually looks like this: https://<server-address>/vmanage-server/rest/rest-api/)')
    parser.add_argument('--user', dest='user', type=str,
                        help='The user name')
    parser.add_argument('--apikey', dest='apikey', type=str,
                        help='The password')
    parser.add_argument('--debug', dest='debug', help='Print debug information about REST requests',
                        action='store_true')
    parser.add_argument('-v', dest='verbose', help='Print exceptions verbosly', action='store_true')
    parser.add_argument('--async', dest='async_param', help='Don\'t wait for commands to finish execution',
                        action='store_true')
    parser.add_argument('--conf', dest='conf', type=str, default=None,
                        help='Configuration file, to avoid settings the parameters in the command line every time. Parameters given on the command line override the configutration file. Default location is .intigua_rest_config in your home folder.')
    parser.add_argument('--profile', '-p', dest='profile', type=str, default="Server",
                        help='Configuration file section - in case you have several servers in the same file. Default is "Server"')
    subparsers = parser.add_subparsers()

    command_subparser = subparsers.add_parser('server-add')
    command_subparser.set_defaults(func=addserver)
    command_subparser.add_argument('--ip', dest='ip', type=str, required=True, help='The server\'s ip')
    command_subparser.add_argument('--name', dest='name', type=str, required=True,
                                   help='The server Intigua interface name')
    command_subparser.add_argument('--dnsname', dest='dnsname', type=str, required=True, help='The server\'s dns name')
    command_subparser.add_argument('--osname', dest='osname', type=str, required=True, help='The OS name')

    command_subparser = subparsers.add_parser('server-merge')
    command_subparser.set_defaults(func=mergeservers)
    command_subparser.add_argument('--fromUrl', dest='fromUrl', type=str, required=True,
                                   help='The server\'s url to merge data from')
    command_subparser.add_argument('--toUrl', dest='toUrl', type=str, required=True,
                                   help='The server\'s url to merge data to')

    command_subparser = subparsers.add_parser('create-config',
                                              help="Create a configuration file from parameters. if --conf specicies the location to write the configuration too, overwriting exiting content. If not provided, the default location is used: " + DEFAULT_CONF_FILE)
    command_subparser.set_defaults(func=createconfig)

    SERVER_HELP = "Identifier of the server you want to remove. free text. To search server using a specific parameter, use a colon. For example: 'ip:10.0.0.1'"

    command_subparser = subparsers.add_parser('server-remove')
    command_subparser.set_defaults(func=removeserver)
    command_subparser.add_argument('server', type=str, help=SERVER_HELP)

    command_subparser = subparsers.add_parser('connector-deploy')
    command_subparser.set_defaults(func=deployconnector)
    command_subparser.add_argument('server', type=str, help=SERVER_HELP)

    command_subparser = subparsers.add_parser('connector-remove')
    command_subparser.set_defaults(func=removeconnector)
    command_subparser.add_argument('server', type=str, help=SERVER_HELP)

    command_subparser = subparsers.add_parser('connector-run-script')
    command_subparser.set_defaults(func=runscript)
    command_subparser.add_argument('server', type=str, help=SERVER_HELP)
    command_subparser.add_argument('script', type=str, help='Local path of the script you wish to run')

    command_subparser = subparsers.add_parser('vagent-apply')
    command_subparser.set_defaults(func=applyvagent)
    command_subparser.add_argument('server', type=str, help=SERVER_HELP)
    command_subparser.add_argument('name', metavar="vagent-name", type=str, help='The vAgent\'s name.')
    command_subparser.add_argument('version', type=str, help='The vAgent\'s version')
    command_subparser.add_argument('package', type=str, help='The name of the configuration package')

    command_subparser = subparsers.add_parser('vagent-start')
    command_subparser.set_defaults(func=startvagent)
    command_subparser.add_argument('server', type=str, help=SERVER_HELP)
    command_subparser.add_argument('name', metavar="vagent-name", type=str, help='The vAgent\'s name.')
    command_subparser.add_argument('version', type=str, help='The vAgent\'s version')

    command_subparser = subparsers.add_parser('vagent-restart')
    command_subparser.set_defaults(func=restartvagent)
    command_subparser.add_argument('server', type=str, help=SERVER_HELP)
    command_subparser.add_argument('name', metavar="vagent-name", type=str, help='The vAgent\'s name.')
    command_subparser.add_argument('version', type=str, help='The vAgent\'s version')

    command_subparser = subparsers.add_parser('vagent-change-config')
    command_subparser.set_defaults(func=changevagentconfig)
    command_subparser.add_argument('server', type=str, help=SERVER_HELP)
    command_subparser.add_argument('name', metavar="vagent-name", type=str, help='The vAgent\'s name.')
    command_subparser.add_argument('version', type=str, help='The vAgent\'s version')
    command_subparser.add_argument('package', type=str, help='The name of the configuration package')

    command_subparser = subparsers.add_parser('vagent-stop')
    command_subparser.set_defaults(func=stopvagent)
    command_subparser.add_argument('server', type=str, help=SERVER_HELP)
    command_subparser.add_argument('name', metavar="vagent-name", type=str, help='The vAgent\'s name.')
    command_subparser.add_argument('version', type=str, help='The vAgent\'s version')

    command_subparser = subparsers.add_parser('vagent-remove')
    command_subparser.set_defaults(func=removevagent)
    command_subparser.add_argument('server', type=str, help=SERVER_HELP)
    command_subparser.add_argument('name', metavar="vagent-name", type=str, help='The vAgent\'s name.')
    command_subparser.add_argument('version', type=str, help='The vAgent\'s version')

    command_subparser = subparsers.add_parser('server-settags')
    command_subparser.set_defaults(func=serversettags)
    command_subparser.add_argument('server', type=str, help=SERVER_HELP)
    command_subparser.add_argument('tags', metavar="tags", nargs='*', type=str, help='The server tags.')

    command_subparser = subparsers.add_parser('server-addtags')
    command_subparser.set_defaults(func=serveraddtags)
    command_subparser.add_argument('server', type=str, help=SERVER_HELP)
    command_subparser.add_argument('tags', metavar="tags", nargs='*', type=str, help='The server tags to add.')

    command_subparser = subparsers.add_parser('server-info')
    command_subparser.set_defaults(func=servergetinfo)
    command_subparser.add_argument('server', type=str, help=SERVER_HELP)

    command_subparser = subparsers.add_parser('vagent-register')
    command_subparser.set_defaults(func=registervagent)
    command_subparser.add_argument('server', type=str, help=SERVER_HELP)
    command_subparser.add_argument('name', metavar="vagent-name", type=str, help='The vAgent\'s name.')
    command_subparser.add_argument('version', type=str, help='The vAgent\'s version')

    command_subparser = subparsers.add_parser('vagent-deregister')
    command_subparser.set_defaults(func=deregistervagent)
    command_subparser.add_argument('server', type=str, help=SERVER_HELP)
    command_subparser.add_argument('name', metavar="vagent-name", type=str, help='The vAgent\'s name.')
    command_subparser.add_argument('version', type=str, help='The vAgent\'s version')

    command_subparser = subparsers.add_parser('vagent-set-groups')
    command_subparser.set_defaults(func=setGroupsVAgent)
    command_subparser.add_argument('server', type=str, help=SERVER_HELP)
    command_subparser.add_argument('name', metavar="vagent-name", type=str, help='The vAgent\'s name.')
    command_subparser.add_argument('version', type=str, help='The vAgent\'s version')

    command_subparser = subparsers.add_parser('define-account')
    command_subparser.set_defaults(func=setServerCreds)
    command_subparser.add_argument('server', type=str, help='Identifier of the server. free text.')
    command_subparser.add_argument('account', metavar="account", type=str, help='The name of the account.')

    command_subparser = subparsers.add_parser('remediate')
    command_subparser.set_defaults(func=remediate)
    command_subparser.add_argument('--ruleName', dest='ruleName', type=str, required=True,
                                   help='The name of the rule that you want to run remediation on')
    command_subparser.add_argument('--serverName', dest='serverName', type=str, default=None, required=False,
                                   help='The server\'s that will be remediate based on ruleName argument')

    args = parser.parse_args()

    try:
        if args.func == createconfig:
            args.func(args)
            return 0

        wasConfigUsed = readConfFile(args)
        if not wasConfigUsed:
            print("You have not used a configuration file to provide parameters. you are strongly suggested to so. use the create-config command to create a config file for you. For Example:")
            print("\t" + sys.argv[0] + " --user=<user> --apikey=<apy-key> --url=<url> create-config")
            print("See " + sys.argv[0] + " --help for more details")

        global client

        if args.debug:
            print("Connecting to", args.user, "@", args.url)

        client = IntiguaRestClient(args.user, args.apikey, args.url, debug=args.debug)

        task_and_etag = args.func(args)
        if (not args.async_param) and task_and_etag:
            task, etag = client.waitForTask(*task_and_etag)
            # if something's returned, it's a task object describing the finished status
            if task["state"] == "completed":
                resource_link = getLink(task, "resource")
                if resource_link:
                    resource = client.doUrlQuery(getLink(task, "resource"))
                    if ("outputMessage" in resource) and ("exitCode" in resource) and (
                                resource["outputMessage"] is not None):
                        print(resource["outputMessage"])
                        return int(resource["exitCode"])
                print("Task completed successfully.")
            else:
                print("Task failed:", task.get("errorData", ""))
                return -1
    except urllib.error.HTTPError as e:
        if args.verbose:
            raise
        if e.getcode() == 403:
            print("Access denied - check user name and API Key")
        elif e.getcode() == 404:
            print("URL Not found - please check that the provided URL is correct")
        else:
            if hasattr(e, "read"):
                try:
                    errData = json.loads(e.read().decode('utf-8'))
                    print(errData["code"] + ":", errData["message"])
                except ValueError:
                    print(e)
            else:
                print(e)
        return -1
    except (ValueError, CmdException) as e:
        print(e)
        return -1


class CmdException(Exception):
    pass


def readConfFile(args):
    # no need to read conf file if we have all the params already..
    if args.url and args.user and args.apikey:
        return False

    confFile = args.conf

    if confFile is not None:
        # if a user provided conf file does not exist - that's an error
        if not os.path.isfile(confFile):
            raise CmdException("Configuration file provided does not exist.")
    else:
        # if we don't even have a default config file - we cant continue, because if we got here, that means that some params are missing.
        confFile = DEFAULT_CONF_FILE
        if not os.path.isfile(confFile):
            raise CmdException(
                "Configuration file not provided, no default configuraiton, and some of the configuration parameters are missing.")

    from configparser import ConfigParser
    config = ConfigParser()
    config.read(confFile)

    get_or_none = lambda configParam: config.get(args.profile, configParam) if config.has_option(args.profile,
                                                                                                 configParam) else None

    args.url = args.url or get_or_none("url")
    args.user = args.user or get_or_none("user")
    args.apikey = args.apikey or get_or_none("apikey")
    return True


def createconfig(args):
    if not (args.url or args.user or args.apikey):
        raise CmdException(
            "Missing configuration, must provide at least one of: url, user, and api key. Preferably all of them")

    if not (args.url and args.user and args.apikey):
        print("You didn't  provided all the configuration parameters (url, user, api key), command line options will be necessary on invocation.")

    confFile = args.conf

    if args.conf is None:
        confFile = DEFAULT_CONF_FILE

    from configparser import ConfigParser
    config = ConfigParser()
    if os.path.isfile(confFile):
        # read exsiting if present, so we won't destroy existing values.
        config.read(confFile)
    if not config.has_section(args.profile):
        config.add_section(args.profile)
    if args.url:
        config.set(args.profile, 'url', args.url)
    if args.user:
        config.set(args.profile, 'user', args.user)
    if args.apikey:
        config.set(args.profile, 'apikey', args.apikey)

    with open(confFile, 'wb') as configfile:
        config.write(configfile)


def getConfigSections(confFile=DEFAULT_CONF_FILE):
    from configparser import ConfigParser
    config = ConfigParser({"debug": "false"})
    config.read(confFile)
    return config.sections()


def getClient(debug=None, profile="Server"):
    from configparser import ConfigParser
    config = ConfigParser({"debug": "false"})
    config.read(DEFAULT_CONF_FILE)
    url = config.get(profile, "url")
    user = config.get(profile, "user")
    apikey = config.get(profile, "apikey")
    if debug is None:
        debug = config.getboolean(profile, "debug")
    return IntiguaRestClient(user, apikey, url, debug=debug, profileName=profile)


def getParamsFromServer(server):
    splittedServer = server.split(":", 1)
    if len(splittedServer) == 2:
        k, v = splittedServer
        return {k: v}
    return None


def getServer(server):
    params = getParamsFromServer(server)
    if not params:
        l = client.findServer(server)
    else:
        l = client.findServerByParams(**params)
    if len(l) != 1:
        raise CmdException(
            "Server search did not return a single server (%d servers found). Please change your server indetifier to be more specific." % len(
                l))
    return l[0]


def removeserver(args):
    s = getServer(args.server)
    task_and_etag = client.removeServer(s)
    return task_and_etag


def addserver(args):
    s = {"ip": args.ip, "name": args.name, "dnsname": args.dnsname, "os": args.osname, "osFullName": args.osname}
    client.addServer(s)


def mergeservers(args):
    servers = {"fromUrl": args.fromUrl, "toUrl": args.toUrl}
    client.mergeServers(servers)


def deployconnector(args):
    s = getServer(args.server)
    task_and_etag = client.deployConnectorToServerTask(s)
    return task_and_etag


def removeconnector(args):
    s = getServer(args.server)
    task_and_etag = client.removeConnectorToServerTask(s)
    return task_and_etag


def runscript(args):
    s = getServer(args.server)
    script = args.script
    task_and_etag = client.runScriptOnServerTask(s, script)
    return task_and_etag


def applyvagent(args):
    s = getServer(args.server)
    task_and_etag = client.deployVAgentToServerTask(s, args.name, args.version, args.package)
    return task_and_etag


def getVagentObject(s, args):
    vagents = client.getVAgentsInServer(s)
    vagents = [v for v in vagents if (v["name"], v["version"]) == (args.name, args.version)]
    if len(vagents) == 0:
        raise CmdException("List of agents on the server doesn't include this vAgent. Is the vAgent deployed ?")
    if len(vagents) > 1:
        raise CmdException(
            "List of matching agents contain more than one element, this should never happen. check parameters.")
    return vagents[0]


def startvagent(args):
    s = getServer(args.server)
    vagent = getVagentObject(s, args)
    task_and_etag = client.startVAgent(vagent)
    return task_and_etag


def restartvagent(args):
    s = getServer(args.server)
    vagent = getVagentObject(s, args)
    task_and_etag = client.restartVAgent(vagent)
    return task_and_etag


def changevagentconfig(args):
    s = getServer(args.server)
    vagent = getVagentObject(s, args)
    task_and_etag = client.changeVAgentConfigPackage(vagent, args.package)
    return task_and_etag


def stopvagent(args):
    s = getServer(args.server)
    vagent = getVagentObject(s, args)
    task_and_etag = client.stopVAgent(vagent)
    return task_and_etag


def removevagent(args):
    s = getServer(args.server)
    vagent = getVagentObject(s, args)
    task_and_etag = client.removeVAgent(vagent)
    return task_and_etag


def serveraddtags(args):
    s = getServer(args.server)
    s["tags"] = [{"name": x} for x in args.tags] + s["tags"]
    client.updateServer(s)


def serversettags(args):
    s = getServer(args.server)
    s["tags"] = [{"name": x} for x in args.tags]
    client.updateServer(s)


def servergetinfo(args):
    s = getServer(args.server)
    print(s)


def registervagent(args):
    s = getServer(args.server)
    vagent = getVagentObject(s, args)
    task_and_etag = client.registerVAgent(vagent)
    return task_and_etag


def deregistervagent(args):
    s = getServer(args.server)
    vagent = getVagentObject(s, args)
    task_and_etag = client.deregisterVAgent(vagent)
    return task_and_etag


def setGroupsVAgent(args):
    s = getServer(args.server)
    vagent = getVagentObject(s, args)
    task_and_etag = client.setGroupsVAgent(vagent)
    return task_and_etag


def setServerCreds(args):
    s = getServer(args.server)
    cred = client.findCredentials(args.account)
    client.defineServerCredentials(s, cred)

def remediate(args):
    client.remediate(args.ruleName, args.serverName)


if __name__ == "__main__":
    sys.exit(main())
